#!/usr/bin/env python3
"""
validate_and_slice_kernels.py
=============================

Inspect each kernel file in the given directory and verify its integrity.
This script performs basic validation on every ``.npy`` file whose name
matches ``kernel_{gauge}_L{L}.npy``.  The checks include:

* The file must be loadable by ``numpy.load``.
* The resulting array must be one‑dimensional and its length must equal
  ``2 * L * L`` as encoded in the filename.
* The array must not contain any NaN values.
* No normalisation is applied; negative values are allowed but noted.

A summary of statistics – minimum, maximum, mean, and standard deviation –
is written to a log file for manual inspection.

Example usage::

    python validate_and_slice_kernels.py --input-dir data/kernels --log logs/kernel_validation.txt

"""

import argparse
import os
import re
import numpy as np


def parse_filename(filename: str) -> tuple[str, int] | None:
    """Return (gauge, L) if filename matches the expected pattern.

    Filenames must be of the form ``kernel_{gauge}_L{L}.npy``.  Gauge
    values are left as strings (e.g. ``SU2``) and L is returned as an
    integer.  If the pattern does not match the filename, None is
    returned.
    """
    match = re.fullmatch(r"kernel_(\w+)_L(\d+)\.npy", filename)
    if match:
        gauge, L_str = match.groups()
        return gauge, int(L_str)
    return None


def main() -> None:
    parser = argparse.ArgumentParser(description="Validate kernel .npy files and record statistics")
    parser.add_argument("--input-dir", required=True, help="Directory containing kernel .npy files")
    parser.add_argument("--log", required=True, help="Path to output log file")
    args = parser.parse_args()

    kernel_dir = args.input_dir
    log_path = args.log

    with open(log_path, "w", encoding="utf-8") as log_file:
        for filename in sorted(os.listdir(kernel_dir)):
            info = parse_filename(filename)
            if info is None:
                continue
            gauge, L = info
            path = os.path.join(kernel_dir, filename)
            try:
                arr = np.load(path)
            except Exception as exc:
                log_file.write(f"{filename}: failed to load – {exc}\n")
                continue

            expected_len = 2 * L * L
            if arr.ndim != 1 or arr.size != expected_len:
                log_file.write(
                    f"{filename}: invalid shape {arr.shape}, expected {expected_len} entries\n"
                )
                continue

            if np.isnan(arr).any():
                log_file.write(f"{filename}: contains NaN values\n")
                continue

            # Compute and record statistics
            min_v = float(arr.min())
            max_v = float(arr.max())
            mean_v = float(arr.mean())
            std_v = float(arr.std())
            log_file.write(
                f"{filename}: min={min_v:.6f}, max={max_v:.6f}, mean={mean_v:.6f}, std={std_v:.6f}\n"
            )

    print(f"Validation complete. Statistics written to {log_path}")


if __name__ == "__main__":  # pragma: no cover
    main()